<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Lms extends ClientsController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('lms_lite_model');
    }

    public function index()
    {
        $data['title'] = _l('lms_course_catalog');
        $data['courses'] = $this->lms_lite_model->get_active_courses();
        $data['wishlist'] = $this->session->userdata('lms_wishlist') ?: [];
        
        $this->data($data);
        $this->view('lms_lite/catalog');
        $this->layout();
    }

    public function course($slug)
    {
        $course = $this->lms_lite_model->get_course_by_slug($slug);
        
        if (!$course || $course['status'] != 'active') {
            show_404();
        }

        $data['course'] = $course;
        $data['lectures'] = $this->lms_lite_model->get_course_lectures($course['id']);
        $data['is_enrolled'] = false;
        
        if (is_client_logged_in()) {
            $data['is_enrolled'] = $this->lms_lite_model->is_client_enrolled(get_contact_user_id(), $course['id']);
        }
        
        $data['title'] = $course['title'];
        
        $this->data($data);
        $this->view('lms_lite/course_detail');
        $this->layout();
    }

    public function enroll($course_id)
    {
        if (!is_client_logged_in()) {
            redirect(site_url('authentication/login'));
        }

        $course = $this->lms_lite_model->get_course($course_id);
        if (!$course) {
            show_404();
        }

        $client_id = get_contact_user_id();
        
        if ($this->lms_lite_model->is_client_enrolled($client_id, $course_id)) {
            set_alert('info', _l('lms_already_enrolled'));
            redirect(site_url('lms/course/' . $course['slug']));
        }

        $enrollment_data = [
            'client_id' => $client_id,
            'course_id' => $course_id,
            'start_date' => date('Y-m-01'),
            'end_date' => date('Y-m-01', strtotime('+' . get_option('lms_lite_default_enrollment_months', 12) . ' months')),
            'status' => 'active'
        ];

        if ($this->lms_lite_model->add_enrollment($enrollment_data)) {
            set_alert('success', _l('lms_enrolled_successfully'));
        } else {
            set_alert('danger', _l('something_went_wrong'));
        }

        redirect(site_url('lms/course/' . $course['slug']));
    }

    public function player($slug, $lecture_id = null)
    {
        if (!is_client_logged_in()) {
            redirect(site_url('authentication/login'));
        }

        $course = $this->lms_lite_model->get_course_by_slug($slug);
        if (!$course) {
            show_404();
        }

        if (!$this->lms_lite_model->is_client_enrolled(get_contact_user_id(), $course['id'])) {
            set_alert('warning', _l('lms_not_enrolled'));
            redirect(site_url('lms/course/' . $slug));
        }

        $lectures = $this->lms_lite_model->get_course_lectures($course['id']);

        if (!$lecture_id && !empty($lectures)) {
            $lecture_id = $lectures[0]['id'];
        }

        $current_lecture = null;
        $prev_lecture = null;
        $next_lecture = null;

        foreach ($lectures as $index => $lecture) {
            if ($lecture['id'] == $lecture_id) {
                $current_lecture = $lecture;
                $prev_lecture = isset($lectures[$index - 1]) ? $lectures[$index - 1] : null;
                $next_lecture = isset($lectures[$index + 1]) ? $lectures[$index + 1] : null;
                break;
            }
        }

        if (!$current_lecture) {
            show_404();
        }

        $data['course'] = $course;
        $data['lectures'] = $lectures;
        $data['current_lecture'] = $current_lecture;
        $data['prev_lecture'] = $prev_lecture;
        $data['next_lecture'] = $next_lecture;
        $data['title'] = $current_lecture['title'];

        $this->data($data);
        $this->view('lms_lite/player');
        $this->layout();
    }

    public function account($section = 'enrollments')
    {
        if (!is_client_logged_in()) {
            redirect(site_url('authentication/login'));
        }

        $client_id = get_contact_user_id();
        
        switch ($section) {
            case 'enrollments':
                $data['enrollments'] = $this->lms_lite_model->get_client_enrollments($client_id);
                $data['title'] = _l('lms_my_enrollments');
                $view = 'lms_lite/account/enrollments';
                break;
                
            case 'orders':
                $data['orders'] = $this->lms_lite_model->get_client_orders($client_id);
                $data['title'] = _l('lms_orders');
                $view = 'lms_lite/account/orders';
                break;
                
            case 'wishlist':
                $wishlist_ids = $this->session->userdata('lms_wishlist') ?: [];
                $data['wishlist_courses'] = $this->lms_lite_model->get_courses_by_ids($wishlist_ids);
                $data['title'] = _l('lms_wishlist');
                $view = 'lms_lite/account/wishlist';
                break;
                
            default:
                show_404();
        }

        $this->data($data);
        $this->view($view);
        $this->layout();
    }

    public function wishlist_toggle()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $course_id = $this->input->post('course_id');
        $wishlist = $this->session->userdata('lms_wishlist') ?: [];

        if (in_array($course_id, $wishlist)) {
            $wishlist = array_diff($wishlist, [$course_id]);
            $action = 'removed';
        } else {
            $wishlist[] = $course_id;
            $action = 'added';
        }

        $this->session->set_userdata('lms_wishlist', array_values($wishlist));

        echo json_encode([
            'success' => true,
            'action' => $action,
            'message' => _l('lms_wishlist_' . $action)
        ]);
    }
}
